/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */

/*
 *Abstrakte Klasse, die eine Nachricht darstellt.
 *eine Nachricht kann z.B. eine SMS oder eine E-Mail sein
 */
package djbuch.kapitel_19;

import java.util.Hashtable;

/**
 * 
 *
 * die abstrakte Klasse Message ist ein loser Rahmen fr die Implementation von Nachrichten.
 * Eine Nachricht kann eine SMS oder eine E-Mail oder eine beliebige andere Implementation sein.
 * die getter und setter dienen dazu, die wichtigsten Parameter (Empfnger, Absender) zu setzen.
 * Jede Implementation muss send () definieren.
 * send () muss die parameter sendCount, sendResponseMessage und sendStatus bedienen.
 * der Hashtable config kann beliebige Konfigurationsparameter aufnehemen (z.B. SMTP Server oder SMS -Gateway)
 *
 * @author Thomas Ekert
 */
public abstract class Message {

	public static final int TYPE_NONE = 0;
	public static final int TYPE_SMS = 1;
	public static final int TYPE_EMAIL = 2;
	public static final int  TYPE_EMAIL_HTML = 3;
	public static final int TYPE_OTHER = 99;
	
	protected int typ=TYPE_NONE;
	private Hashtable config = null;
	protected String sender = null;
	private String recipient = null;
	protected String sendResponseMessage = null;
	protected boolean sendStatus = false;
	protected int sendCount = 0;
	protected String message = null;
	
	/**
	 * Basis Konstruktor
	 * @param type - Typ der Message. Vordefiniert sind SMS und EMAIL
	 * @param configuration - Ein Hashtable zur freien Verwendung
	 * @throws MessageException
	 */
	protected Message(int type, Hashtable configuration) throws MessageException {
			if (type <= TYPE_NONE) {
				throw new MessageException ("Message kann nur mit type > " + TYPE_NONE + " erstellt werden.");
			}
			this.typ = type;
			this.config = configuration==null?new Hashtable():configuration;
	}

	/**
	 * 
	 * @return - null oder Wert
	 */
	public String getRecipient() {
		return recipient;
	}
	
	/**
	 * 
	 * @return - null oder Wert
	 */
	public String getSender() {
		return sender;
	}
	
	/**
	 * @param sender - text, der als Absender angezeigt wird. Sollte den angegebenen Sender auf plausibilitt prfen.
	 */
	abstract void setSender(String sender) throws MessageException;

	
	/**
	 * Setzt den Empfnger der Message. Je nach implementierung ist dies ein fr die Implementierung geeigneter String (also. z.B.
	 * eine Telefonnummer fr SMS oder E-Mail Adresse fr E-Mail
	 * @param rec - Empfnger
	 * @throws MessageException - falls der zu Setzende Recipient nicht ok (hierfr muss isRecipientOK implementiert werden.) ist.
	 */
	public void setRecipient (String rec) throws MessageException{
		if (! isRecipientOK(rec)) {
			throw new MessageException ("Recipient darf nicht null sein.");
		}
		this.recipient = rec;
	}
	
	/** Validierung fr zulssige Recipients
	 * 
	 * @param rec
	 * @return
	 */
	abstract boolean isRecipientOK (String rec);
	
	public Hashtable getConfigurations () {
		return this.config;
	}
	
	/**
	 * liefert einen frei definierten Parameter der unter key im Hashtable abgespeichert ist.
	 * @param key
	 * @return
	 * @throws MessageException
	 */
	public Object getConfiguration (String key) throws MessageException {
		if (key==null || key.equals("")) {
			throw new MessageException ("Key darf nicht null sein.");
		}
		return this.config.get(key);
	}
	
	/** setzt einen Parameter im Hashtable 
	 * 
	 * @param key
	 * @param value
	 */
	public void setConfiguration (String key, Object value) {
		this.config.put(key, value);
	}
	
	/**
	 * 
	 * @return - liefert null (default) oder die bei dem Versenden der Message erzeugte Antwort vom Message-Gateway
	 * Muss bei der Implementierung im send() gesetzt werden.
	 */
	public String getResponseMessage () {
		return this.sendResponseMessage;
	}
	
	/**
	 * 
	 * @return - status (Erfolg) des Message Versands. Muss bei der Implementierung im send() gesetzt werden.
	 */
	public boolean getSendStatus () {
		return this.sendStatus;
	}
	
	/**
	 * 
	 * @return anzahl der Versuche eine Nachricht zu versenden.
	 * Muss bei der Implementierung im send() gesetzt werden.
	 */
	public int getSendCount () {
		return this.sendCount;
	}
	
	/**
	 * muss die Nachricht auf zulssigkeit prfen und dann in die lokale var message setzen.
	 * @param message
	 * @throws MessageException
	 */
	abstract void setMessage (String message) throws MessageException;

	/** liefert die zuvor gesetzte zu verschickende Nachricht
	 * 
	 * @return
	 */
	public String getMessage () {
		return this.message;
	}
	
	/**
	 * Fhrt das Verschicken der Nachricht durch. Muss sendCount, sendReturnMessage und sendStatus bedienen.
	 * 
	 * @return - erfolg des Versands.
	 * @throws MessageException
	 */
	abstract boolean send() throws MessageException;
	
	public static String stringMailType (int mailType) {
		if (mailType == TYPE_EMAIL) {
			return "PlainText";
		} else if (mailType == TYPE_EMAIL_HTML) {
			return "HTML";
		} else if (mailType == TYPE_SMS) {
			return "SMS";
		} else {
			return "OTHER";
		}
	}
}
